<?php

namespace App\Http\Controllers;

use App\Helpers\SmsHelper;
use App\Helpers\SteadfastCourier;
use App\Mail\InvoiceEmailManager;
use App\Models\Address;
use App\Models\Cart;
use App\Models\CombinedOrder;
use App\Models\Coupon;
use App\Models\CouponUsage;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\OrdersExport;
use App\Models\OrderStatus;
use App\Models\Product;
use App\Models\SmsTemplate;

use App\Models\User;
use App\Notifications\OrderNotification;
use App\Utility\EmailUtility;
use App\Utility\NotificationUtility;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Route;
use Maatwebsite\Excel\Facades\Excel;
use CoreComponentRepository;

class WholeSaleController extends Controller
{
    public function __construct()
    {
        // Staff Permission Check
        $this->middleware(['permission:view_all_orders|view_inhouse_orders|view_seller_orders|view_pickup_point_orders|view_all_offline_payment_orders'])->only('all_orders');
        $this->middleware(['permission:view_order_details'])->only('show');
        $this->middleware(['permission:delete_order'])->only('destroy','bulk_order_delete');
    }

    // All Orders
    public function all_orders(Request $request)
    {

        CoreComponentRepository::instantiateShopRepository();

        $date = $request->date;
        $sort_search = null;
        $delivery_status = null;
        $payment_status = '';
        $order_type = '';

        $orders = Order::where('order_type','wholesale')->orderBy('id', 'desc');
        $admin_user_id = get_admin()->id;

//        if (Route::currentRouteName() == 'inhouse_orders.index' && Auth::user()->can('view_inhouse_orders')) {
//            $orders = $orders->where('orders.seller_id', '=', $admin_user_id);
//        }
//        elseif (Route::currentRouteName() == 'seller_orders.index' && Auth::user()->can('view_seller_orders')) {
//            $orders = $orders->where('orders.seller_id', '!=', $admin_user_id);
//        }
//        elseif (Route::currentRouteName() == 'pick_up_point.index' && Auth::user()->can('view_pickup_point_orders')) {
//            if (get_setting('vendor_system_activation') != 1) {
//                $orders = $orders->where('orders.seller_id', '=', $admin_user_id);
//            }
//            $orders->where('shipping_type', 'pickup_point')->orderBy('code', 'desc');
//            if (
//                Auth::user()->user_type == 'staff' &&
//                Auth::user()->staff->pick_up_point != null
//            ) {
//                $orders->where('shipping_type', 'pickup_point')
//                    ->where('pickup_point_id', Auth::user()->staff->pick_up_point->id);
//            }
//        }
//        elseif (Route::currentRouteName() == 'all_orders.index' && Auth::user()->can('view_all_orders')) {
//            if (get_setting('vendor_system_activation') != 1) {
//                $orders = $orders->where('orders.seller_id', '=', $admin_user_id);
//            }
//        }
//        elseif (Route::currentRouteName() == 'offline_payment_orders.index' && Auth::user()->can('view_all_offline_payment_orders')) {
//            $orders = $orders->where('orders.manual_payment', 1);
//            if($request->order_type != null){
//                $order_type = $request->order_type;
//                $orders = $order_type =='inhouse_orders' ?
//                    $orders->where('orders.seller_id', '=', $admin_user_id) :
//                    $orders->where('orders.seller_id', '!=', $admin_user_id);
//            }
//        }
//        elseif (Route::currentRouteName() == 'unpaid_orders.index' && Auth::user()->can('view_all_unpaid_orders')) {
//            $orders = $orders->where('orders.payment_status', 'unpaid');
//        }
//        else {
//            abort(403);
//        }

        if ($request->search) {
            $sort_search = $request->search;
            $orders = $orders->where('code', 'like', '%' . $sort_search . '%');
        }
        if ($request->payment_status != null) {
            $orders = $orders->where('payment_status', $request->payment_status);
            $payment_status = $request->payment_status;
        }
        if ($request->delivery_status != null) {
            $orders = $orders->where('delivery_status', $request->delivery_status);
            $delivery_status = $request->delivery_status;
        }
        if ($date != null) {
            $orders = $orders->where('created_at', '>=', date('Y-m-d', strtotime(explode(" to ", $date)[0])) . '  00:00:00')
                ->where('created_at', '<=', date('Y-m-d', strtotime(explode(" to ", $date)[1])) . '  23:59:59');
        }
        $orders = $orders->paginate(15);


        $unpaid_order_payment_notification = get_notification_type('complete_unpaid_order_payment', 'type');
        return view('backend.whole_sales.index', compact('orders', 'sort_search', 'order_type', 'payment_status', 'delivery_status', 'date', 'unpaid_order_payment_notification'));
    }

    public function show($id)
    {
        $order = Order::findOrFail(decrypt($id));

        $order_shipping_address = json_decode($order->shipping_address);
        $delivery_boys = User::where('city', $order_shipping_address->city)
            ->where('user_type', 'delivery_boy')
            ->get();

        if(env('DEMO_MODE') != 'On') {
            $order->viewed = 1;
            $order->save();
        }

        return view('backend.sales.show', compact('order', 'delivery_boys'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create_all_orders()
    {
        $data['products'] = Product::with('thumbnail')->get();
        $data['customers'] = User::where('user_type', 'customer')->get();
//         dd($data['products']);
        return view('backend.whole_sales.create_order',$data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $carts = Cart::where('user_id', Auth::user()->id)->active()->get();

        if ($carts->isEmpty()) {
            flash(translate('Your cart is empty'))->warning();
            return redirect()->route('home');
        }

        $address = Address::where('id', $carts[0]['address_id'])->first();

        $shippingAddress = [];
        if ($address != null) {
            $shippingAddress['name']        = Auth::user()->name;
            $shippingAddress['email']       = Auth::user()->email;
            $shippingAddress['address']     = $address->address;
            $shippingAddress['country']     = $address->country->name;
            $shippingAddress['state']       = $address->state->name;
            $shippingAddress['city']        = $address->city->name;
            $shippingAddress['postal_code'] = $address->postal_code;
            $shippingAddress['phone']       = $address->phone;
            if ($address->latitude || $address->longitude) {
                $shippingAddress['lat_lang'] = $address->latitude . ',' . $address->longitude;
            }
        }

        $combined_order = new CombinedOrder;
        $combined_order->user_id = Auth::user()->id;
        $combined_order->shipping_address = json_encode($shippingAddress);
        $combined_order->save();

        $seller_products = array();
        foreach ($carts as $cartItem) {
            $product_ids = array();
            $product = Product::find($cartItem['product_id']);
            if (isset($seller_products[$product->user_id])) {
                $product_ids = $seller_products[$product->user_id];
            }
            array_push($product_ids, $cartItem);
            $seller_products[$product->user_id] = $product_ids;
        }

        $firstOrderCode = null;

        foreach ($seller_products as $seller_product) {
            $order = new Order;
            $order->combined_order_id = $combined_order->id;
            $order->user_id = Auth::user()->id;
            $order->shipping_address = $combined_order->shipping_address;
            $order->additional_info = $request->additional_info;
            $order->payment_type = $request->payment_option;
            $order->delivery_viewed = '0';
            $order->payment_status_viewed = '0';
            $order->code = date('Ymd-His') . rand(10, 99);
            $order->date = strtotime('now');
            $order->save();

            if ($firstOrderCode === null) {
                $firstOrderCode = $order->code; // capture first order code
            }

            $subtotal = 0;
            $tax = 0;
            $shipping = 0;
            $coupon_discount = 0;

            //Order Details Storing
            foreach ($seller_product as $cartItem) {
                $product = Product::find($cartItem['product_id']);

                $subtotal += cart_product_price($cartItem, $product, false, false) * $cartItem['quantity'];
                $tax +=  cart_product_tax($cartItem, $product, false) * $cartItem['quantity'];
                $coupon_discount += $cartItem['discount'];

                $product_variation = $cartItem['variation'];

                $product_stock = $product->stocks->where('variant', $product_variation)->first();
                if ($product->digital != 1 && $cartItem['quantity'] > $product_stock->qty) {
                    flash(translate('The requested quantity is not available for ') . $product->getTranslation('name'))->warning();
                    $order->delete();
                    return redirect()->route('cart')->send();
                } elseif ($product->digital != 1) {
                    $product_stock->qty -= $cartItem['quantity'];
                    $product_stock->save();
                }

                $order_detail = new OrderDetail;
                $order_detail->order_id = $order->id;
                $order_detail->seller_id = $product->user_id;
                $order_detail->product_id = $product->id;
                $order_detail->variation = $product_variation;
                $order_detail->price = cart_product_price($cartItem, $product, false, false) * $cartItem['quantity'];
                $order_detail->tax = cart_product_tax($cartItem, $product, false) * $cartItem['quantity'];
                $order_detail->shipping_type = $cartItem['shipping_type'];
                $order_detail->product_referral_code = $cartItem['product_referral_code'];
                $order_detail->shipping_cost = $cartItem['shipping_cost'];

                $shipping += $order_detail->shipping_cost;
                //End of storing shipping cost

                $order_detail->quantity = $cartItem['quantity'];

                if (addon_is_activated('club_point')) {
                    $order_detail->earn_point = $product->earn_point;
                }

                $order_detail->save();

                $product->num_of_sale += $cartItem['quantity'];
                $product->save();

                $order->seller_id = $product->user_id;
                $order->shipping_type = $cartItem['shipping_type'];

                if ($cartItem['shipping_type'] == 'pickup_point') {
                    $order->pickup_point_id = $cartItem['pickup_point'];
                }
                if ($cartItem['shipping_type'] == 'carrier') {
                    $order->carrier_id = $cartItem['carrier_id'];
                }

                if ($product->added_by == 'seller' && $product->user->seller != null) {
                    $seller = $product->user->seller;
                    $seller->num_of_sale += $cartItem['quantity'];
                    $seller->save();
                }

                if (addon_is_activated('affiliate_system')) {
                    if ($order_detail->product_referral_code) {
                        $referred_by_user = User::where('referral_code', $order_detail->product_referral_code)->first();

                        $affiliateController = new AffiliateController;
                        $affiliateController->processAffiliateStats($referred_by_user->id, 0, $order_detail->quantity, 0, 0);
                    }
                }
            }

            $order->grand_total = $subtotal + $tax + $shipping;

            if ($seller_product[0]->coupon_code != null) {
                $order->coupon_discount = $coupon_discount;
                $order->grand_total -= $coupon_discount;

                $coupon_usage = new CouponUsage;
                $coupon_usage->user_id = Auth::user()->id;
                $coupon_usage->coupon_id = Coupon::where('code', $seller_product[0]->coupon_code)->first()->id;
                $coupon_usage->save();
            }

            $combined_order->grand_total += $order->grand_total;

            $order->save();
        }

        $combined_order->save();

        $customerPhone = $shippingAddress['phone'] ?? null;

        if ($customerPhone && $firstOrderCode) {
            $smsMessage = "Dear {$shippingAddress['name']}, your order #{$firstOrderCode} has been placed successfully. Thank you for shopping with Jersey Factory! For any query please call 01971359319";
            SmsHelper::send($customerPhone, $smsMessage);
        }

        $request->session()->put('combined_order_id', $combined_order->id);
    }


    public function store_manual_order(Request $request)
    {
        $request->validate([
            'customer_name' => 'nullable|string|max:255',
            'customer_phone' => 'nullable|string|max:20',
            'shipping_address' => 'nullable|string',
            'city' => 'nullable|string',
            'country' => 'nullable|string',
            'payment_method' => 'nullable|string',
            'delivery_status' => 'nullable|string',
            'payment_status' => 'nullable|string',
            'shipping_cost' => 'nullable|numeric|min:0',
            'coupon_discount' => 'nullable|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // Find or create customer
            $customer = User::where('phone', $request->customer_phone)->first();
            if (!$customer) {
                $customer = User::create([
                    'name' => $request->customer_name,
                    'phone' => $request->customer_phone,
                    'email' => $request->customer_phone . '@manual.order',
                    'user_type' => 'customer',
                ]);
            }

            // Prepare shipping address
            $shippingAddress = [
                'name' => $request->customer_name,
                'email' => $customer->email,
                'phone' => $request->customer_phone,
                'address' => $request->shipping_address,
                'country' => $request->country,
                'state' => $request->state ?? '',
                'city' => $request->city,
                'postal_code' => $request->postal_code ?? '',
            ];

            // Create Combined Order
            $combined_order = CombinedOrder::create([
                'user_id' => $customer->id,
                'shipping_address' => json_encode($shippingAddress),
            ]);

            // Determine seller from first product
            $firstProductId = $request->product_ids[0] ?? null;
            if (!$firstProductId) {
                throw new \Exception("No products provided for the order.");
            }
            $firstProduct = Product::findOrFail($firstProductId);
            $seller_id = $firstProduct->user_id;

            // Create single order
            $order = Order::create([
                'combined_order_id' => $combined_order->id,
                'user_id' => $customer->id,
                'seller_id' => $seller_id,
                'shipping_address' => json_encode($shippingAddress),
                'additional_info' => $request->order_note ?? '',
                'payment_type' => $request->payment_method ?? null,
                'payment_status' => $request->payment_status ?? 'unpaid',
                'delivery_status' => $request->delivery_status ?? 'pending',
                'shipping_cost' => $request->shipping_cost ?? 0,
                'coupon_discount' => $request->coupon_discount ?? 0,
                'order_type' =>'wholesale',
                'delivery_viewed' => '0',
                'payment_status_viewed' => '0',
                'code' => date('Ymd-His') . rand(10, 99),
                'date' => strtotime('now'),
                'shipping_type' => 'flat_rate',
            ]);

            $subtotal = 0;
            $total_tax = 0;

            // Loop through products
            foreach ($request->product_ids as $key => $productId) {
                if (!$productId) continue; // Skip empty rows

                $product = Product::findOrFail($productId);

                $quantity = $request->quantities[$key] ?? 1;
                $unit_price = $request->whole_sale_prices[$key] ?? 0;
                $old_price = $request->prices[$key] ?? 0;
                $tax = $request->taxes[$key] ?? 0;
                $shipping = $request->shipping_costs[$key] ?? 0;

                $item_subtotal = $unit_price * $quantity;
                $item_tax = ($item_subtotal * $tax) / 100;

                $subtotal += $item_subtotal;
                $total_tax += $item_tax;

                // Insert or update order detail
                OrderDetail::updateOrCreate(
                    ['id' => $request->order_detail_ids[$key] ?? 0],
                    [
                        'order_id' => $order->id,
                        'seller_id' => $product->user_id,
                        'product_id' => $product->id,
                        'price' => $unit_price,
                        'whole_sale_prices' => $old_price,
                        'tax' => $item_tax,
                        'shipping_cost' => $shipping,
                        'quantity' => $quantity,
                    ]
                );

                // Update stock for non-digital products
                if ($product->digital != 1) {
                    $product->current_stock -= $quantity;
                    $product->num_of_sale += $quantity;
                    $product->save();
                }

                // Update seller stats if needed
                if ($product->added_by == 'seller' && $product->user->seller) {
                    $seller = $product->user->seller;
                    $seller->num_of_sale += $quantity;
                    $seller->save();
                }
            }

            // Calculate grand total
            $shipping_cost = $request->shipping_cost ?? 0;
            $discount = $request->coupon_discount ?? 0;
            $order->grand_total = $subtotal + $total_tax + $shipping_cost - $discount;
            $order->save();

            $combined_order->grand_total = $order->grand_total;
            $combined_order->save();

            DB::commit();

            // Send SMS notification
            try {
                if ($request->customer_phone && $order->code) {
                    $smsMessage = "Dear {$shippingAddress['name']}, your order #{$order->code} has been placed successfully.";
                    SmsHelper::send($request->customer_phone, $smsMessage);
                }
            } catch (\Exception $e) {
                \Log::error('Failed to send SMS: ' . $e->getMessage());
            }

            // Send notification to customer
            try {
                Notification::send($customer, new OrderNotification($order));
            } catch (\Exception $e) {
                \Log::error('Failed to send order notification: ' . $e->getMessage());
            }

            flash(translate('Order has been placed successfully'))->success();
            return redirect()->route('whole_sale_all_orders.index');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Manual order creation failed: ' . $e->getMessage());
            flash(translate('Failed to create order: ') . $e->getMessage())->error();
            return back()->withInput();
        }
    }
    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */


    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $order = Order::with('orderDetails.product')->findOrFail($id);
        $products =Product::get();
//dd($order);
        return view('backend.whole_sales.edit_order', compact('order','products'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id) {
//    dd($request->all());
        $order = Order::findOrFail($id);

        // Update shipping address
        $shippingAddress = [
            'name' => $request->shipping_name,
            'email' => $request->shipping_email,
            'phone' => $request->shipping_phone,
            'address' => $request->shipping_address,
            'city' => $request->shipping_city,
            'state' => $request->shipping_state,
            'postal_code' => $request->shipping_postal_code,
            'country' => $request->shipping_country,
        ];

        // Calculate grand total
        $subtotal = 0;
        foreach ($request->quantities as $key => $quantity) {
            $price = $request->whole_sale_prices[$key];
            $tax = $request->taxes[$key] ?? 0;
            $shipping = $request->shipping_costs[$key] ?? 0;
            $subtotal += ($quantity * $price) + $tax + $shipping;
        }
        $grandTotal = ($subtotal + $request->shipping_cost ?? 0) - ($request->coupon_discount ?? 0);

        // Update order
        $order->update([
            'shipping_address' => json_encode($shippingAddress),
            'shipping_type' => $request->shipping_type,
            'delivery_status' => $request->delivery_status,
            'payment_status' => $request->payment_status,
            'payment_type' => $request->payment_type,
            'payment_details' => $request->payment_details,
            'tracking_code' => $request->tracking_code,
            'steadfast_tracking_code' => $request->steadfast_tracking_code,
            'steadfast_consignment_id' => $request->steadfast_consignment_id,
            'pathao_consignment_id' => $request->pathao_consignment_id,
            'pathao_status' => $request->pathao_status,
            'additional_info' => $request->additional_info,
            'coupon_discount' => $request->coupon_discount ?? 0,
            'shipping_cost' => $request->shipping_cost ?? 0,
            'grand_total' => $grandTotal,
        ]);

        // Update order details (product IDs remain unchanged)

        $currentDetailIds = [];

        foreach ($request->product_ids ?? [] as $key => $productId) {
            $detailId = $request->order_detail_ids[$key] ?? null;

            $detailData = [
                'order_id' => $order->id,
                'product_id' => $productId,
                'quantity' => $request->quantities[$key] ?? 0,
                'price' => $request->whole_sale_prices[$key] * $request->quantities[$key] ?? 0,
                'whole_sale_prices' => $request->prices[$key] * $request->quantities[$key] ?? 0,
                'tax' => $request->taxes[$key] ?? 0,
                'shipping_cost' => $request->shipping_costs[$key] ?? 0,
            ];

            if ($detailId) {
                // Update existing
                OrderDetail::where('id', $detailId)->update($detailData);
                $currentDetailIds[] = $detailId;
            } else {
                // Insert new
                $newDetail = OrderDetail::create($detailData);
                $currentDetailIds[] = $newDetail->id;
            }
        }

        // Delete any order_details not in the request
        OrderDetail::where('order_id', $order->id)
            ->whereNotIn('id', $currentDetailIds)
            ->delete();


        return redirect()->route('whole_sale_all_orders.index')->with('success', 'Order updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $order = Order::findOrFail($id);
        if ($order != null) {
            $order->commissionHistory()->delete();
            foreach ($order->orderDetails as $key => $orderDetail) {
                try {
                    product_restock($orderDetail);
                } catch (\Exception $e) {
                }

                $orderDetail->delete();
            }
            $order->delete();
            flash(translate('Order has been deleted successfully'))->success();
        } else {
            flash(translate('Something went wrong'))->error();
        }
        return back();
    }

    public function bulk_order_delete(Request $request)
    {
        if ($request->id) {
            foreach ($request->id as $order_id) {
                $this->destroy($order_id);
            }
        }

        return 1;
    }

    public function order_details(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->save();
        return view('seller.order_details_seller', compact('order'));
    }

    public function update_delivery_status(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->delivery_viewed = '0';
        $order->delivery_status = $request->status;
        $order->save();

        if($order->save()){
            $orderStatus = new OrderStatus();
            $orderStatus->status = $request->status;
            $orderStatus->order_id = $order->id;
            $orderStatus->save();
        }

        if($request->status == 'delivered'){
            $order->delivered_date = date("Y-m-d H:i:s");
            $order->save();
        }

        if ($request->status == 'cancelled' && $order->payment_type == 'wallet') {
            $user = User::where('id', $order->user_id)->first();
            $user->balance += $order->grand_total;
            $user->save();
        }

        // If the order is cancelled and the seller commission is calculated, deduct seller earning
        if($request->status == 'cancelled' && $order->user->user_type == 'seller' && $order->payment_status == 'paid' && $order->commission_calculated == 1){
            $sellerEarning = $order->commissionHistory->seller_earning;
            $shop = $order->shop;
            $shop->admin_to_pay -= $sellerEarning;
            $shop->save();
        }

        if (Auth::user()->user_type == 'seller') {
            foreach ($order->orderDetails->where('seller_id', Auth::user()->id) as $key => $orderDetail) {
                $orderDetail->delivery_status = $request->status;
                $orderDetail->save();

                if ($request->status == 'cancelled') {
                    product_restock($orderDetail);
                }
            }
        } else {
            foreach ($order->orderDetails as $key => $orderDetail) {

                $orderDetail->delivery_status = $request->status;
                $orderDetail->save();

                if ($request->status == 'cancelled') {
                    product_restock($orderDetail);
                }

                if (addon_is_activated('affiliate_system')) {
                    if (($request->status == 'delivered' || $request->status == 'cancelled') &&
                        $orderDetail->product_referral_code
                    ) {

                        $no_of_delivered = 0;
                        $no_of_canceled = 0;

                        if ($request->status == 'delivered') {
                            $no_of_delivered = $orderDetail->quantity;
                        }
                        if ($request->status == 'cancelled') {
                            $no_of_canceled = $orderDetail->quantity;
                        }

                        $referred_by_user = User::where('referral_code', $orderDetail->product_referral_code)->first();

                        $affiliateController = new AffiliateController;
                        $affiliateController->processAffiliateStats($referred_by_user->id, 0, 0, $no_of_delivered, $no_of_canceled);
                    }
                }
            }
        }
        // Delivery Status change email notification to Admin, seller, Customer
        EmailUtility::order_email($order, $request->status);

        // Delivery Status change SMS notification
        if (addon_is_activated('otp_system') && SmsTemplate::where('identifier', 'delivery_status_change')->first()->status == 1) {
            try {
                SmsUtility::delivery_status_change(json_decode($order->shipping_address)->phone, $order);
            } catch (\Exception $e) {}
        }

        //Send web Notifications to user
        NotificationUtility::sendNotification($order, $request->status);

        //Sends Firebase Notifications to user
        if (get_setting('google_firebase') == 1 && $order->user->device_token != null) {
            $request->device_token = $order->user->device_token;
            $request->title = "Order updated !";
            $status = str_replace("_", "", $order->delivery_status);
            $request->text = " Your order {$order->code} has been {$status}";

            $request->type = "order";
            $request->id = $order->id;
            $request->user_id = $order->user->id;

            NotificationUtility::sendFirebaseNotification($request);
        }


//        if (addon_is_activated('delivery_boy')) {
//            if (Auth::user()->user_type == 'delivery_boy') {
//                $deliveryBoyController = new DeliveryBoyController;
//                $deliveryBoyController->store_delivery_history($order);
//            }
//        }

        return 1;
    }

    public function update_tracking_code(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->tracking_code = $request->tracking_code;
        $order->save();

        return 1;
    }

    public function update_payment_status(Request $request)
    {
        $order = Order::findOrFail($request->order_id);
        $order->payment_status_viewed = '0';
        $order->save();

        if (Auth::user()->user_type == 'seller') {
            foreach ($order->orderDetails->where('seller_id', Auth::user()->id) as $key => $orderDetail) {
                $orderDetail->payment_status = $request->status;
                $orderDetail->save();
            }
        } else {
            foreach ($order->orderDetails as $key => $orderDetail) {
                $orderDetail->payment_status = $request->status;
                $orderDetail->save();
            }
        }

        $status = 'paid';
        foreach ($order->orderDetails as $key => $orderDetail) {
            if ($orderDetail->payment_status != 'paid') {
                $status = 'unpaid';
            }
        }
        $order->payment_status = $status;
        $order->save();


        if (
            $order->payment_status == 'paid' &&
            $order->commission_calculated == 0
        ) {
            calculateCommissionAffilationClubPoint($order);
        }

        // Payment Status change email notification to Admin, seller, Customer
        if($request->status == 'paid'){
            EmailUtility::order_email($order, $request->status);
        }

        //Sends Web Notifications to Admin, seller, Customer
        NotificationUtility::sendNotification($order, $request->status);

        //Sends Firebase Notifications to Admin, seller, Customer
        if (get_setting('google_firebase') == 1 && $order->user->device_token != null) {
            $request->device_token = $order->user->device_token;
            $request->title = "Order updated !";
            $status = str_replace("_", "", $order->payment_status);
            $request->text = " Your order {$order->code} has been {$status}";

            $request->type = "order";
            $request->id = $order->id;
            $request->user_id = $order->user->id;

            NotificationUtility::sendFirebaseNotification($request);
        }


        if (addon_is_activated('otp_system') && SmsTemplate::where('identifier', 'payment_status_change')->first()->status == 1) {
            try {
                SmsUtility::payment_status_change(json_decode($order->shipping_address)->phone, $order);
            } catch (\Exception $e) {
            }
        }
        return 1;
    }

    public function assign_delivery_boy(Request $request)
    {
        if (addon_is_activated('delivery_boy')) {

            $order = Order::findOrFail($request->order_id);
            $order->assign_delivery_boy = $request->delivery_boy;
            $order->delivery_history_date = date("Y-m-d H:i:s");
            $order->save();

            $delivery_history = \App\Models\DeliveryHistory::where('order_id', $order->id)
                ->where('delivery_status', $order->delivery_status)
                ->first();

            if (empty($delivery_history)) {
                $delivery_history = new \App\Models\DeliveryHistory;

                $delivery_history->order_id = $order->id;
                $delivery_history->delivery_status = $order->delivery_status;
                $delivery_history->payment_type = $order->payment_type;
            }
            $delivery_history->delivery_boy_id = $request->delivery_boy;

            $delivery_history->save();

            if (env('MAIL_USERNAME') != null && get_setting('delivery_boy_mail_notification') == '1') {
                $array['view'] = 'emails.invoice';
                $array['subject'] = translate('You are assigned to delivery an order. Order code') . ' - ' . $order->code;
                $array['from'] = env('MAIL_FROM_ADDRESS');
                $array['order'] = $order;

                try {
                    Mail::to($order->delivery_boy->email)->queue(new InvoiceEmailManager($array));
                } catch (\Exception $e) {
                }
            }

            if (addon_is_activated('otp_system') && SmsTemplate::where('identifier', 'assign_delivery_boy')->first()->status == 1) {
                try {
                    SmsUtility::assign_delivery_boy($order->delivery_boy->phone, $order->code);
                } catch (\Exception $e) {
                }
            }
        }

        return 1;
    }

    public function orderBulkExport(Request $request)
    {
        if($request->id){
            return Excel::download(new OrdersExport($request->id), 'orders.xlsx');
        }
        return back();
    }

    public function unpaid_order_payment_notification_send(Request $request){
        if($request->order_ids != null){
            $notificationType = get_notification_type('complete_unpaid_order_payment', 'type');
            foreach (explode(",",$request->order_ids) as $order_id) {
                $order = Order::where('id', $order_id)->first();
                $user = $order->user;
                if($notificationType->status == 1 && $order->payment_status == 'unpaid'){
                    $order_notification['order_id']     = $order->id;
                    $order_notification['order_code']   = $order->code;
                    $order_notification['user_id']      = $order->user_id;
                    $order_notification['seller_id']    = $order->seller_id;
                    $order_notification['status']       = $order->payment_status;
                    $order_notification['notification_type_id'] = $notificationType->id;
                    Notification::send($user, new OrderNotification($order_notification));
                }
            }
            flash(translate('Notification Sent Successfully.'))->success();
        }
        else{
            flash(translate('Something went wrong!.'))->warning();
        }
        return back();
    }

    //Send to SteadFast courier for pickup
    public function sendToCourier($id)
    {
        $order = Order::findOrFail($id);

        // Prevent duplicate sending
        if ($order->steadfast_consignment_id) {
            return back()->with('error', 'Order already sent to courier.');
        }

        $steadfast = new SteadfastCourier();
        $response = $steadfast->createOrder($order);

        if (isset($response['status']) && $response['status'] == 200) {
            $order->steadfast_consignment_id = $response['consignment']['consignment_id'] ?? null;
            $order->steadfast_tracking_code  = $response['consignment']['tracking_code'] ?? null;
            $order->save();

            return back()->with('success', 'Order successfully sent to courier.');
        }

        return back()->with('error', $response['message'] ?? 'Failed to send order to courier.');
    }
}
